/**
 * Basic usage:
 * ```ts
 * let familyTree = new FamilyTree2(document.getElementById('tree'));
 * familyTree.readOnly = false;
 * familyTree.addFamilyMembers([
 *    {id: 1, spouseIds: [2], motherId: 3, fatherId: 4},
 *    {id: 2},
 *    {id: 3},
 *    {id: 4},
 * ]).draw(1);
 * ```
 *
 * Load on demand usage
 * ```ts
 * let familyTree = new FamilyTree2(document.getElementById('tree'));
 * familyTree.readOnly = true;
 * familyTree.templateName = "kitkat";
 *
 * familyTree.onDemand(function (args) {
 *     fetch(`https://familytree.balkan.app/FetchFamilyMemebers/c8sshc/${args.ids.join(',')}`)
 *         .then((data) => data.json())
 *         .then((data) => familyTree.addFamilyMembers(data).draw(null, null, familyTree.fit))
 * });
 *
 * familyTree.onFocusButtonClick(function (args) {
 *     fetch('https://familytree.balkan.app/FetchFamily/c8sshc/' + args.newFocusId)
 *         .then((data) => data.json())
 *         .then((data) => familyTree.addFamilyMembers(data).draw(args.newFocusId, null, familyTree.fit));
 *     return false;
 * });
 *
 * fetch('https://familytree.balkan.app/FetchFamily/c8sshc/Q60772')
 *     .then((data) => data.json())
 *     .then((data) => familyTree.addFamilyMembers(data).draw('Q60772'));
 * ```
 */
declare class FamilyTree2 {
    /**
     * Internal property for storing chart data and configurations.
     * @ignore
     */
    _: any;
    /**
     * Gets the SVG element of the chart.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * const svg = familyTree.svgElement;
     * ```
     */
    get svgElement(): SVGElement;
    get shortcuts(): {
        [key: string]: {
            keysPressed?: Array<string>;
            mouseActions?: Array<string>;
            activeComponentType?: string;
            desc?: string;
        };
    };
    /**
     * Gets or sets the read-only state of the chart.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.readOnly = false;
     * const readOnly = familyTree.readOnly;
     * ```
     */
    get readOnly(): boolean;
    set readOnly(value: boolean);
    /**
     * Gets or sets the visibility of the navigation bar.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.navigationBar = false;
     * const navBar = familyTree.navigationBar;
     * ```
     */
    get navigationBar(): boolean;
    set navigationBar(value: boolean);
    /**
     * Gets or sets the current mode of the chart. It could be 'dark' or 'light'
     * Default value: "light";
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.mode = "dark";
     * ```
     */
    get mode(): 'dark' | 'light' | string;
    set mode(value: "dark" | "light" | string);
    /**
     * The collapsed node IDs.
     * This property allows you to get and set the collapsed node IDs.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * let collapsedIds = familyTree.collapsedIds;  // Get the collapsed node IDs
     * familyTree.collapsedIds = [6, 8, 9, 12];    // Set the collapsed node IDs
     * ```
     */
    get collapsedIds(): Array<string | number>;
    set collapsedIds(value: Array<string | number>);
    /**
     * Gets the edited node ID.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.readOnly = false;
     *
     * familyTree.addFamilyMembers([{id: 1}])
     *   .draw(1, null, function(){
     *      this.edit(1, 'name');
     *      console.log(this.editId)
     *   });
     * ```
     */
    get editId(): string | number;
    /**
     * Gets the root HTML element of the chart.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * let treeElement = familyTree.element;
     * ```
     */
    get element(): HTMLElement;
    /**
     * Persist the state (scale, position, expand/collapse and focus) in the url or local storage
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familytree.state.name = "StateForMyTree";
     * familyTree.state.writeToLocalStorage = true;
     * familyTree.state.readFromLocalStorage = true;
     * familyTree.state.writeToSessionStorage = true;
     * familyTree.state.readFromSessionStorage = true;
     * familyTree.state.writeToUrlParams = true;
     * familyTree.state.readFromUrlParams = true;
     * ```
     */
    state: {
        name: string;
        readFromUrlParams: boolean;
        writeToUrlParams: boolean;
        readFromLocalStorage: boolean;
        writeToLocalStorage: boolean;
        readFromSessionStorage: boolean;
        writeToSessionStorage: boolean;
    };
    /**
     * Gets the focus ID.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([{ id: 1, spouseIds: [2] }, { id: 2 }])
     *    .draw(1, null, function() {
     *        let focusId = familyTree.focusId;
     *    });
     * ```
     */
    get focusId(): string | number;
    /**
     * Gets the family data.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([{ id: 1, spouseIds: [2] }, { id: 2 }]).draw(1);
     *
     * let family = familyTree.family;
     * ```
     */
    get family(): Array<FamilyTree2.FamilyMember>;
    /**
     * Gets or sets the template name.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.templateName = 'kitkat';
     * let templateName = familyTree.templateName;
     * ```
     */
    get templateName(): string;
    set templateName(value: string);
    /**
     * Gets the template object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * let template = familyTree.template;
     * ```
     */
    get template(): FamilyTree2.Template;
    /**
     * @ignore
     */
    autoGenerateSiblingChildSpouseIds: boolean;
    /**
     * Initializes a new instance of FamilyTree2.
     *
     * @param element The HTML element where the family tree will be rendered.
     *
     * @example
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * ```
     */
    constructor(element?: HTMLElement | string);
    /**
     * Clears the chart.
     *
     * ```ts
     * familyTree.onLoaded(() => {
     *    familyTree.clear()
     * });
     * ```
     */
    clear(): void;
    /**
     * Show node relations to add.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.readOnly = false;
     *
     * familyTree.addFamilyMembers([{ id: 1 }])
     *    .draw(1, null, function() {
     *        this.edit(1, 'name');
     *    });
     * ```
     */
    edit(familyMemberId: string | number, callback?: (this: FamilyTree2) => void): void;
    /**
     * Retrieves an SVG representation of the chart.
     * @returns SVG as string
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([{ id: 1, spouseIds: [2] }, { id: 2 }])
     *    .draw(1);
     *
     * let svg = familyTree.svg();
     * ```
     */
    svg(): string;
    /**
     * Retrieves a family member by ID.
     * @param id The family member's ID.
     * @returns Femily member object.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     *
     * let familyMember = familyTree.getFemilyMember(2);
     * ```
     */
    getFemilyMember(id: string | number): FamilyTree2.FamilyMember;
    /**
     * Returns the node object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onLoaded(() => {
     *    let node = familyTree.getNode(2);
     *    //...
     * });
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, spouseIds: [2], childIds: [3, 4] },
     *    { id: 2, childIds: [3, 4] },
     *    { id: 3 },
     *    { id: 4 }
     * ]).draw(1);
     * ```
     */
    getNode(id: string | number): FamilyTree2.Node;
    centerNodes(nodes: Array<FamilyTree2.Node>): void;
    static convertGedcomToFamilyMembers(gedcomText: string): {
        familyMembers: Array<FamilyTree2.FamilyMember>;
        fistIndividualId: string | number;
    };
    exportToGedcom(): string;
    center(node: FamilyTree2.Node): void;
    /**
     * Fits the family tree in the tree div.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    {
     *        id: 1,
     *        fatherId: 2,
     *        motherId: 3,
     *        spouseIds: [4],
     *        siblingIds: [5, 6],
     *        childIds: [7],
     *        name: 'Me'
     *    },
     *    { id: 2, name: 'Father' },
     *    { id: 3, name: 'Mother' },
     *    { id: 4, childIds: [7], name: 'Spouse' },
     *    { id: 5, name: 'Sibling 1', motherId: 8 },
     *    { id: 6, name: 'Sibling 2' },
     *    { id: 7, name: 'Child' },
     *    { id: 8, name: 'Mother 1' }
     * ]).draw(1);
     *
     * familyTree.fit();
     * ```
     */
    fit(): void;
    /**
     * @ignore
     * @param nodeIds
     * @param callback
     */
    makeNodesVisible(nodeIds: Array<string | number>, callback?: (this: FamilyTree2) => void): void;
    /**
     * Draws the tree.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     * ```
     */
    draw(focusId?: string | number, freezeId?: string | number, callback?: (this: FamilyTree2) => void): void;
    /**
     * Removes specified family member
     * @param familyMemberId Family member id that will be removed
     * @returns Family tree object
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]);
     *
     * familyTree.remove(2).draw(1);
     * ```
     */
    remove(familyMemberId: string | number): FamilyTree2;
    /**
     * Add family members.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     * ```
     */
    addFamilyMembers(familyMembers: Array<FamilyTree2.FamilyMember>): FamilyTree2;
    /**
     * Adds a father to the specified family member.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ].addFather(1, {name: "Derek"}).draw(1);
     * ```
     * @param childId The ID of the family member to whom the father will be added.
     * @param familyMember Tha family member object
     * @returns Family tree object.
     */
    addFather(childId: string | number, familyMember?: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Adds a mother to the specified family member.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addMother(1, {name: "Maria"}).draw(1);
     * ```
     * @param childId The ID of the family member to whom the mother will be added.
     * @param familyMember The data object containing information about the mother.
     * @returns Family tree object.
     */
    addMother(childId: string | number, familyMember?: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Adds a sibling to the specified family member.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addSibling(1, {name: "Maria"}).draw(1);
     * ```
     * @param siblingId The ID of the family member to whom the sibling will be added.
     * @param familyMember The data of the new sibling.
     * @returns Family tree object.
     */
    addSibling(siblingId: string | number, familyMember?: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Adds a spouse to the specified family member.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addSpouse(1, {name: "Maria"}).draw(1);
     * ```
     * @param spouseId The ID of the family member to whom the spouse will be added.
     * @param familyMember The spouse's data.
     * @returns Family tree object.
     */
    addSpouse(spouseId: string | number, familyMember: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Adds an unmarried partner to the specified family member.
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addUnmarriedPartner(1, {name: "Maria"}).draw(1);
     * ```
     * @param familyMemberId The ID of the family member to whom the partner will be added.
     * @param familyMember he data of the unmarried partner to be added.
     * @returns Family tree object.
     */
    addUnmarriedPartner(familyMemberId: string | number, familyMember: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Adds a child to the family tree.
     *
     * Here is an example with two parents of the added child:
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addChild(1, 2, {name: "Maria"}).draw(1);
     * ```
     * Here is an example with one parent of the added child:
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addChild(1, {name: "Maria"}).draw(1);
     * ```
     * @param parent1Id The ID of the first parent.
     * @param parent2Id The ID of the second parent (optional).
     * @param familyMember Additional data for the child node (optional).
     * @returns Family tree object.
     */
    addChild(parent1Id: string | number, parent2Id?: string | number, familyMember?: FamilyTree2.FamilyMember): FamilyTree2;
    /**
     * Select family member node.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1, null, function () {
     *    this.select(2);
     * });
     * ```
     */
    select(familyMemberId: string | number): FamilyTree2;
    /**
     * Shows the edit/details form for a node ID.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     *
     * familyTree.showEditor(2);
     * ```
     */
    showEditor(familyMemberId: string | number): FamilyTree2;
    /**
     * @ignore
     * @returns
     */
    hideEditor(): FamilyTree2;
    /**
     * Destroys the familyTree instance.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     *
     * familyTree.destroy();
     * ```
     */
    destroy(): void;
    /**
     * Clears the state
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.clearState('myStateName');
     * ```
     */
    clearState(stateName: string): void;
    /**
     * Creates URL parameners from the state
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     * familytree.state.name = "StateForMyTree";
     * familyTree.state.writeToUrlParams = true;
     * familyTree.state.readFromUrlParams = true;
     *
     * document.getElementById('button').addEventListener('click', function () {
     *    let location = window.location + '?' + familyTree.stateToUrl()
     *    window.open(location);
     * });
     * ```
     */
    stateToUrl(): string;
    /**
     * Registers an event listener for the 'demand' event.
     * Triggers when new data from the server is required.
     *
     * @param listener The function to be called when the 'demand' event is triggered.
     * @returns Family tree object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onDemand(function (args) {
     *   fetch(`https://familytree.balkan.app/FetchFamilyMembers/c8sshc/${args.ids.join(',')}`)
     *       .then((data) => data.json())
     *       .then((data) => familyTree.addFamilyMembers(data).draw());
     * });
     * ```
     */
    onDemand(listener: (args: {
        /**
         * An array of IDs that needs to be pulled from the server or any source.
         */
        ids: Array<string | number>;
    }) => void): FamilyTree2;
    /**
     * Registers an event listener for the onFocusButtonClick event.
     * Triggers when a new family member is focused from the UI or programmatically.
     *
     * @param listener The function to be called when the onFocusButtonClick event is triggered.
     * @returns Family tree object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onFocusButtonClick(function (args) {
     *    fetch('https://familytree.balkan.app/FetchFamily/c8sshc/' + args.newFocusId)
     *        .then((data) => data.json())
     *        .then((data) => familyTree.addFamilyMembers(data).draw(args.newFocusId, familyTree.fit));
     *    return false;
     * });
     * ```
     */
    onFocusButtonClick(listener: (args: {
        /**
         * New family tree member ID that will be focussed
         */
        newFocusId: string | number;
        /**
         * Old family tree member ID that was focussed
         */
        oldFocusId: string | number;
    }) => void): FamilyTree2;
    /**
     * Registers an update listener for the 'updated' event.
     * Triggers when one or more family tree members are updated, added, or removed.
     *
     * @param listener The function to be called when the 'updated' event is triggered.
     * @returns Family tree.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onUpdated(function (args) {
     *     // Option 1 - update your server with the new data in args
     *     // Option 2 - update your server with the array from familyTree.family
     * });
     * ```
     */
    onUpdated(listener: (args: {
        /**
         * An array of IDs that needs to be removed or delted from the storage.
         */
        remove: Array<string | number>;
        /**
         * An array of family members that needs to be updated / synced in the storage
         */
        update: Array<FamilyTree2.FamilyMember>;
        /**
         * An array of family members that needs to be inserted or added in the storage
         */
        insert: Array<FamilyTree2.FamilyMember>;
    }) => void): FamilyTree2;
    /**
     * Fires an event when you click on a node
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onNodeClick(function(args){
     *    alert('node clicked')
     * });
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     * ```
     * @param listener
     * @returns
     */
    onNodeClick(listener: (args: {
        node: FamilyTree2.Node;
        event: MouseEvent;
    }) => void): FamilyTree2;
    /**
     * Fires an event when you click outside a anode
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onSvgClick(function(args){
     *    alert('svg clicked')
     * });
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).draw(1);
     * ```
     * @param listener
     * @returns
     */
    onSvgClick(listener: (args: {
        event: MouseEvent;
    }) => void): FamilyTree2;
    /**
     * Registers a new event listener that will be called after the family tree has been loaded.
     *
     * @param listener The function to be called after the tree is loaded.
     * @returns Family tree object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onLoaded(() => {
     *    alert('Family tree loaded');
     * });
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2] },
     *    { id: 2, name: "Ana" }
     * ]).addSpouse(1, { name: "Maria" }).draw(1);
     * ```
     */
    onLoaded(listener: (args: {}) => void): FamilyTree2;
    /**
     * Creates a family tree template.
     *
     * @param targetTemplateName The name of the new template.
     * @param sourceTemplateName The name of the source template.
     * @returns Family tree template object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.template = FamilyTree2.createTemplate("myTemplate", "base");
     * ```
     */
    static createTemplate(targetTemplateName: string, sourceTemplateName?: string): FamilyTree2.Template;
}
declare namespace FamilyTree2 {
    /**
     * @ignore
     */
    interface Point {
        x: number;
        y: number;
    }
    /**
     * The Node object.
     *
     * Basic usage:
     * ```ts
     * let node = familyTree.getNode(2);
     * ```
     *
     * Full example:
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.onLoaded(() => {
     *    let node = familyTree.getNode(2);
     *    //...
     * });
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, spouseIds: [2], childIds: [3, 4] },
     *    { id: 2, childIds: [3, 4] },
     *    { id: 3 },
     *    { id: 4 }
     * ]).draw(1);
     * ```
     */
    interface Node {
        /**
         * The id of the node
         *
         * ```ts
         * let id = node.id;
         * ```
         */
        readonly id: string | number;
        /**
         * The template id of the node (depending on the relationship type)
         *
         * ```ts
         * let templateId = node.templateId;
         * ```
         */
        templateId: string;
        /**
         * The left X point of the node.
         *
         * ```ts
         * let left = node.left;
         * ```
         */
        readonly left: number;
        /**
         * The right X point of the node.
         *
         * ```ts
         * let right = node.right;
         * ```
         */
        readonly right: number;
        /**
         * The top Y point of the node.
         *
         * ```ts
         * let top = node.top;
         * ```
         */
        readonly top: number;
        /**
         * The bottom Y point of the node.
         *
         * ```ts
         * let bottom = node.bottom;
         * ```
         */
        readonly bottom: number;
        /**
         * The width of the node.
         *
         * ```ts
         * let width = node.width;
         * ```
         */
        width: number;
        /**
         * The height of the node.
         *
         * ```ts
         * let height = node.height;
         * ```
         */
        height: number;
        /**
         * @ignore
         */
        type: string;
        /**
         * True if the node is selected
         *
         * ```ts
         * let selected = node.selected;
         * ```
         */
        selected: boolean;
        /**
         * The HTML element of the node
         *
         * ```ts
         * let element = node.element;
         * ```
         */
        element: HTMLElement;
        /**
         * The stroke color of the node
         *
         * ```ts
         * let stroke = node.stroke;
         * ```
         */
        stroke: string;
        /**
         * The stroke width of the node
         *
         * ```ts
         * let strokeWidth = node.strokeWidth;
         * ```
         */
        strokeWidth: number;
        /**
         * @ignore
         */
        [name: string]: any;
        /**
         * @ignore
         */
        layer: number;
        /**
         * The family tree data of the node
         *
         * ```ts
         * let familyMember = node.familyMember;
         * ```
         */
        familyMember: FamilyTree2.FamilyMember;
        /**
         * The relationship type of the node
         *
         * ```ts
         * let relationshipType = node.relationshipType;
         * ```
         */
        relationshipType: string;
        /**
         * The node level
         *
         * ```ts
         * let level = node.level;
         * ```
         */
        level: number;
        /**
         * The collpased parens of the node
         *
         * ```ts
         * let collapsedParents = node.collapsedParents;
         * ```
         */
        collapsedParents: Array<number | string>;
        /**
         * @ignore
         */
        primaryParentId: number | string;
        /**
         * @ignore
         */
        secondaryParentId: number | string;
        /** Node parents
         *
         * ```ts
         * let parents = node.parents;
         * ```
         */
        parents: Array<number | string>;
        /** Node mother id
         *
         * ```ts
         * let motherId = node.motherId;
         * ```
         */
        motherId: number | string;
        /** Node father id
         *
         * ```ts
         * let fatherId = node.fatherId;
         * ```
         */
        fatherId: number | string;
        /** Node children ids
         *
         * ```ts
         * let children = node.children;
         * ```
         */
        children: Array<number | string>;
        /** Collpased children ids
         *
         * ```ts
         * let collapsedChildren = node.collapsedChildren;
         * ```
         */
        collapsedChildren: Array<number | string>;
        /** Collpased siblings ids
         *
         * ```ts
         * let collapsedSiblings = node.collapsedSiblings;
         * ```
         */
        collapsedSiblings: Array<number | string>;
        /** Node siblings ids
         *
         * ```ts
         * let siblings = node.siblings;
         * ```
         */
        siblings: Array<number | string>;
        /** Node sibling id
         * @ignore
         */
        sibling: number | string;
        /** Collpased spouses ids
         *
         * ```ts
         * let collapsedSpouses = node.collapsedSpouses;
         * ```
         */
        collapsedSpouses: Array<number | string>;
        /** Node spouses ids
         *
         * ```ts
         * let spouses = node.spouses;
         * ```
         */
        spouses: Array<number | string>;
        /** Node sibling id
         * @ignore
         */
        spouse: number | string;
        /** True if it is unmarried partner
         *
         * ```ts
         * let isUnmarriedPartner = node.isUnmarriedPartner;
         * ```
         */
        isUnmarriedPartner: boolean;
        /** True if it is step parent
         *
         * ```ts
         * let isStepParent = node.isStepParent;
         * ```
         */
        isStepParent: boolean;
        /** The index number of the spouse with children
         *
         * ```ts
         * let spouseWithChildrenIndex = node.spouseWithChildrenIndex;
         * ```
         */
        spouseWithChildrenIndex: number;
    }
    /**
     * @ignore
     */
    interface Link {
        readonly width: number;
        readonly height: number;
        readonly length: number;
        readonly pathElement: HTMLElement;
        templateId: string;
        points: Array<Point>;
        from: number | string;
        to: number | string;
        fromPort: number | string;
        toPort: number | string;
    }
    /**
     * @ignore
     */
    interface ShapeTemplate {
        displayName: string;
        svg(link: FamilyTree2.Node): string;
        html(link: FamilyTree2.Node): string;
        ports(link: FamilyTree2.Node): {
            [key: string]: {
                x: number;
                y: number;
            };
        };
        offset(shape: FamilyTree2.Node): {
            x: number;
            y: number;
        };
        svgFront(link: FamilyTree2.Node): string;
        width?: number | string;
        height?: number | string;
        minWidth?: number;
        minHeight?: number;
        fill?: string;
        stroke?: string;
        strokeWidth?: number;
        displayInShapeBar?: boolean;
        displayInPortShapeBar?: boolean;
        resizable?: boolean;
        static?: boolean;
    }
    /**
     * The template link object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * let link = familyTree.template.link;
     * ```
     */
    interface LinkTemplate {
        /**
         *
         * @ignore
         */
        path(link: FamilyTree2.Link): string;
        /**
         * The template link object.
         *
         * @example
         * ```ts
         * let familyTree = new FamilyTree2(document.getElementById('tree'));
         *
         * familyTree.template.link.svg = function (link) {
         *    let strokeDasharray = '';
         *    let stroke = link.stroke;
         *
         *    if (link.toNode.familyMember.flag) {
         *        strokeDasharray = 'stroke-dasharray="4"';
         *        stroke = 'blue';
         *    }
         *
         *    return `<path ${strokeDasharray} style="fill: none; stroke: ${stroke}; stroke-width: ${link.strokeWidth}px;" marker-start="url(#rounded_start)" marker-end="url(#rounded_end)"></path>`;
         * };
         * ```
         */
        svg(link: FamilyTree2.Link): string;
        /**
         * The template link stroke color.
         *
         * @example
         * ```ts
         * let familyTree = new FamilyTree2(document.getElementById('tree'));
         *
         * familyTree.template.link.stroke = "red";
         * ```
         */
        stroke?: string;
        /**
         * The template link stroke width.
         *
         * @example
         * ```ts
         * let familyTree = new FamilyTree2(document.getElementById('tree'));
         *
         * familyTree.template.link.strokeWidth = 3;
         * ```
         */
        strokeWidth?: number;
        /**
         * @ignore
         */
        markerStart?: string;
        /**
         * @ignore
         */
        markerEnd?: string;
        /**
         * @ignore
         */
        markerMid?: string;
    }
    /**
     * A family member object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * familyTree.addFamilyMembers([
     *    { id: 1, name: "John", spouseIds: [2], address: "London" },
     *    { id: 2, name: "Ana", address: "London" }
     * ]).draw(1);
     *
     * let familyMember = familyTree.getFamilyMember();
     * ```
     */
    interface FamilyMember {
        /**
         * ```ts
         * let id = familymember.id
         * ```
         */
        id: string | number;
        /**
         * ```ts
         * let motherId = familymember.motherId;
         * ```
         */
        motherId?: string | number;
        /**
         * ```ts
         * let fatherId = familymember.fatherId;
         * ```
         */
        fatherId?: string | number;
        /**
         * ```ts
         * let siblingIds = familymember.siblingIds;
         * ```
         */
        siblingIds?: Array<string | number>;
        /**
         * ```ts
         * let spouseIds = familymember.spouseIds;
         * ```
         */
        spouseIds?: Array<string | number>;
        /**
         * ```ts
         * let childIds = familymember.childIds;
         * ```
         */
        childIds?: Array<string | number>;
        /**
         * ```ts
         * let unmarriedPartnerIds = familymember.unmarriedPartnerIds;
         * ```
         */
        unmarriedPartnerIds?: Array<string | number>;
        /**
         * ```ts
         * let stepParentIds = familymember.stepParentIds;
         * ```
         */
        stepParentIds?: Array<string | number>;
        /**
         * ```ts
         * let sexOrGender = familymember.sexOrGender;
         * ```
         */
        sexOrGender?: 'male' | 'female' | string;
        /**
         * ```ts
         * let name = familymember.name;
         * let address = familymember.address;
         * ```
         */
        [name: string]: any;
    }
    /**
     * A template object.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * let template = familyTree.template;
     * ```
     */
    interface Template {
        /**
         * Defines he node HTML.
         *
         * ```ts
         * template.html = function(node) {
         *   return `<div data-field="address">${node.familyMember.name}</div>`;
         * };
         * ```
         */
        html(value: (node: FamilyTree2.Node) => void): any;
        /**
         * Defines he node SVG.
         *
         * ```ts
         * template.svg = function(node){
         *   return `<rect rx="20" ry="20" x="0" y="0"
         *              width="${node.width}" height="${node.height}"
         *              style=" stroke:${node.stroke}; fill: ${node.fill}; stroke-width: ${node.strokeWidth};">
         *          </rect>`;
         * };
         * ```
         */
        svg(value: (node: FamilyTree2.Node) => void): any;
        /**
         * Adds SVG to the node.
         *
         * ```ts
         * template.insertSvg = function(node){
         *   return `<text text-anchor="end" x="${node.width}" y="${node.height + 16}">
         *              ${node.relationshipType}
         *          </text>`
         * };
         * ```
         */
        insertSvg(value: (node: FamilyTree2.Node) => void): any;
        /**
         * Defines the SVG buttons.
         *
         * ```ts
         * familyTree.template.svgButton = function (node, x, y, text, icon, type, color, title) {
         *     let radius = 12;
         *
         *     if (type == 'edit' || type == 'camera') {
         *         return '';
         *     }
         *
         *     if (type == 'focus') {
         *         radius = 16;
         *         x = 0;
         *         y = node.height / 2;
         *         icon = `
         *         <svg xmlns="http://www.w3.org/2000/svg"
         *              width="24" height="24" x="4" y="4"
         *              viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2">
         *             <circle cx="11" cy="11" r="8"></circle>
         *             <line x1="21" y1="21" x2="16.65" y2="16.65"></line>
         *         </svg>`;
         *     }
         *
         *     return `
         *         <g transform="translate(${x - radius}, ${y - radius})"
         *            data-button="${type}" class="bft2-button">
         *             <circle cx="${radius}" cy="${radius}" r="${radius}"
         *                     style="fill:${color}; stroke-width:0;" />
         *             ${icon}
         *             ${text}
         *         </g>`;
         * }
         * ```
         */
        svgButton(node: FamilyTree2.Node, x: number, y: number, text: string, icon: string, type: string, color: string, title: string): string;
        /**
         * Nodes width.
         *
         * ```ts
         * template.width = 300;
         * template.nodes.focus.width = 350;
         * ```
         */
        width: number;
        /**
         * Nodes height.
         *
         * ```ts
         * template.height = 200;
         * template.nodes.focus.height = 250;
         * ```
         */
        height: number;
        /**
         * Nodes color.
         *
         * ```ts
         * template.fill = '#F57C00';
         * template.nodes.focus.fill = '#039BE5';
         * ```
         */
        fill: string;
        /**
         * Nodes border width.
         *
         * ```ts
         * template.strokeWidth = 5;
         * template.nodes.focus.strokeWidth = 7;
         * ```
         */
        strokeWidth: number;
        /**
         * Nodes border color.
         *
         * ```ts
         * template.stroke = '#FFCA28';
         * template.nodes.focus.stroke = '#F57C00';
         * ```
         */
        stroke: number;
        /**
         * @ignore
         */
        editor: {
            header(node: FamilyTree2.Node): string;
            buttons(node: FamilyTree2.Node): string;
            content(node: FamilyTree2.Node, family: Array<FamilyTree2.FamilyMember>): string;
            readOnlyContent(node: FamilyTree2.Node, family: Array<FamilyTree2.FamilyMember>): string;
        };
        /**
         * Relationship node types
         *
         * ```ts
         * let nodes = template.nodes;
         * ```
         */
        nodes: {
            focus: FamilyTree2.ShapeTemplate;
            parent: FamilyTree2.ShapeTemplate;
            spouse: FamilyTree2.ShapeTemplate;
            sibling: FamilyTree2.ShapeTemplate;
            child: FamilyTree2.ShapeTemplate;
            parentInLaw: FamilyTree2.ShapeTemplate;
            parentSibling: FamilyTree2.ShapeTemplate;
            stepParent: FamilyTree2.ShapeTemplate;
            childInLaw: FamilyTree2.ShapeTemplate;
            pibling: FamilyTree2.ShapeTemplate;
        };
        /**
         * Template Link
         *
         * ```ts
         * let link = template.link;
         * ```
         */
        link: FamilyTree2.LinkTemplate;
        /**
         * Separations between nodes
         *
         * ```ts
         * let separations = template.separations;
         * ```
         */
        separations: FamilyTree2.Separations;
        /**
         * @ignore
         */
        svgVacant(node: FamilyTree2.ShapeTemplate): string;
        /**
         * @ignore
         */
        defs: string;
    }
    /**
     * Separations between nodes.
     *
     * ```ts
     * let familyTree = new FamilyTree2(document.getElementById('tree'));
     *
     * let separations = familyTree.template.separations;
     * ```
     */
    interface Separations {
        /**
         * Separations between nodes.
         *
         * ```ts
         * separations.parentInLawLevel = 220;
         * ```
         */
        parentInLawLevel: number;
        /**
         * ```ts
         * separations.parentInLawLevel = 180;
         * ```
         */
        parentSiblingLevel: number;
        /**
         * ```ts
         * separations.childSubtree = 180;
         * ```
         */
        childSubtree: number;
        /**
         * ```ts
         * separations.childLevel = 150;
         * ```
         */
        childLevel: number;
        /**
         * ```ts
         * separations.childNeighbor = 200;
         * ```
         */
        childNeighbor: number;
        /**
         * ```ts
         * separations.parentSubtree = 200;
         * ```
         */
        parentSubtree: number;
        /**
         * ```ts
         * separations.parentLevel = 200;
         * ```
         */
        parentLevel: number;
        /**
         * ```ts
         * separations.parentNeighbor = 220;
         * ```
         */
        parentNeighbor: number;
        /**
         * ```ts
         * separations.focusSide = 200;
         * ```
         */
        focusSide: number;
        /**
         * ```ts
         * separations.focusBottom = 150;
         * ```
         */
        focusBottom: number;
        /**
         * ```ts
         * separations.childSide = 200;
         * ```
         */
        childSide: number;
        /**
         * ```ts
         * separations.parentSide = 200;
         * ```
         */
        parentSide: number;
        /**
         * ```ts
         * separations.piblingNeighbor = 90;
         * ```
         */
        piblingNeighbor: number;
        /**
         * ```ts
         * separations.siblingNeighbor = 90;
         * ```
         */
        siblingNeighbor: number;
        /**
         * ```ts
         * separations.childInLowNeighbor = 90;
         * ```
         */
        childInLowNeighbor: number;
        /**
         * ```ts
         * separations.spouseNeighbor = 90;
         * ```
         */
        spouseNeighbor: number;
        /**
         * ```ts
         * separations.stepParentNeighbor = 90;
         * ```
         */
        stepParentNeighbor: number;
        /**
         * ```ts
         * separations.parentInLawNeighbor = 90;
         * ```
         */
        parentInLawNeighbor: number;
        /**
         * ```ts
         * separations.noneNeighbor = 90;
         * ```
         */
        parentSiblingNeighbor: number;
    }
}
